/* -*-  Mode: C++; c-file-style: "gnu"; indent-tabs-mode:nil; -*- */
/*
 * Author: Yu Cao <caoyu08@csnet1.cs.tsinghua.edu.cn>
 */


#define NS_LOG_APPEND_CONTEXT \
  if (m_node) { std::clog << Simulator::Now ().GetSeconds () << " [node " << m_node->GetId () << "] "; }

#include "nampt-cc.h"
#include "ns3/log.h"
#include "ns3/trace-source-accessor.h"
#include "ns3/simulator.h"
#include "ns3/abort.h"
#include "ns3/node.h"

NS_LOG_COMPONENT_DEFINE ("NaMPTCC");

namespace ns3 {

NS_OBJECT_ENSURE_REGISTERED (NaMPTCC);

TypeId
NaMPTCC::GetTypeId (void)
{
  static TypeId tid = TypeId ("ns3::NaMPTCC")
    .SetParent<TcpSocketBase> ()
    .AddConstructor<NaMPTCC> ()
    .AddTraceSource ("CongestionWindow",
                     "The TCP connection's congestion window",
                     MakeTraceSourceAccessor (&NaMPTCC::m_cWnd))
  ;
  return tid;
}

NaMPTCC::NaMPTCC (void) 
	: m_inFastRec (false)
{
  NS_LOG_FUNCTION (this);
}

NaMPTCC::NaMPTCC (const NaMPTCC& sock)
  : TcpSocketBase (sock),
    m_cWnd (sock.m_cWnd),
    m_ssThresh (sock.m_ssThresh),
    m_initialCWnd (sock.m_initialCWnd),
    m_inFastRec (false)
{
  NS_LOG_FUNCTION (this);
  NS_LOG_LOGIC ("Invoked the copy constructor");
}

NaMPTCC::NaMPTCC (const TcpSocketBase& sock)
  : TcpSocketBase (sock),
    m_inFastRec (false)
{
}

NaMPTCC::~NaMPTCC (void)
{
}

int
NaMPTCC::Listen (void)
{
  NS_LOG_FUNCTION (this);
  InitializeCwnd ();
  return TcpSocketBase::Listen ();
}

/** Limit the size of in-flight data by cwnd and receiver's rxwin */
uint32_t
NaMPTCC::Window (void)
{
  NS_LOG_FUNCTION (this);
  return std::min (m_rWnd.Get (), m_cWnd.Get () );
}

Ptr<TcpSocketBase>
NaMPTCC::Fork (void)
{
  return CopyObject<NaMPTCC> (this);
}

/** New ACK (up to seqnum seq) received. Increase cwnd and call TcpSocketBase::NewAck() */
void
NaMPTCC::NewAck (const SequenceNumber32& seq)
{
  NS_LOG_FUNCTION (this << seq);
  NS_LOG_LOGIC ("NaMPTCC receieved ACK for seq " << seq <<
                " cwnd " << m_cWnd <<
                " ssthresh " << m_ssThresh);

  // Check for exit condition of fast recovery
  if (m_inFastRec)
    { // RFC2001, sec.4; RFC2581, sec.3.2
      // First new ACK after fast recovery: reset cwnd
      m_cWnd = m_ssThresh;
      m_inFastRec = false;
      NS_LOG_INFO ("Reset cwnd to " << m_cWnd);
    };

  // Notification
  bool hasDone = false;
  if (!m_notifyNewAck.IsNull ())
  {
    hasDone = m_notifyNewAck (this, seq);
  }

  // default process
  if (!hasDone)
  {
    // Increase of cwnd based on current phase (slow start or congestion avoidance)
    if (m_cWnd < m_ssThresh)
      { // Slow start mode, add one segSize to cWnd. Default m_ssThresh is 65535. (RFC2001, sec.1)
        m_cWnd += m_segmentSize;
        NS_LOG_INFO ("In SlowStart, updated to cwnd " << m_cWnd << " ssthresh " << m_ssThresh);
      }
    else
      { // Congestion avoidance mode, increase by (segSize*segSize)/cwnd. (RFC2581, sec.3.1)
        // To increase cwnd for one segSize per RTT, it should be (ackBytes*segSize)/cwnd
        double adder = static_cast<double> (m_segmentSize * m_segmentSize) / m_cWnd.Get ();
        adder = std::max (1.0, adder);
        m_cWnd += static_cast<uint32_t> (adder);
        NS_LOG_INFO ("In CongAvoid, updated to cwnd " << m_cWnd << " ssthresh " << m_ssThresh);
      }
  }
  
  // Complete newAck processing
  TcpSocketBase::NewAck (seq);
}

// Fast recovery and fast retransmit
void
NaMPTCC::DupAck (const TcpHeader& t, uint32_t count)
{
  NS_LOG_FUNCTION (this << "t " << count);

  // Notification
  bool hasDone = false;
  if (!m_notifyDupAck.IsNull ())
  {
    hasDone = m_notifyDupAck (this, t.GetAckNumber(), count);
  }

  // default process
  if (!hasDone)
  {
    if (count == 3 && !m_inFastRec)
    { // triple duplicate ack triggers fast retransmit (RFC2581, sec.3.2)
      m_ssThresh = std::max (2 * m_segmentSize, BytesInFlight () / 2);
      m_cWnd = m_ssThresh + 3 * m_segmentSize;
    }
    else if (m_inFastRec)
    { // In fast recovery, inc cwnd for every additional dupack (RFC2581, sec.3.2)
      m_cWnd += m_segmentSize;
    }
  }

  // common process
  if (count == 3 && !m_inFastRec)
  { // triple duplicate ack triggers fast retransmit (RFC2581, sec.3.2)
    m_inFastRec = true;
    NS_LOG_INFO ("Triple dupack. Reset cwnd to " << m_cWnd << ", ssthresh to " << m_ssThresh);
    DoRetransmit ();
  }
  else if (m_inFastRec)
  { // In fast recovery, inc cwnd for every additional dupack (RFC2581, sec.3.2)
    NS_LOG_INFO ("Increased cwnd to " << m_cWnd);
    SendPendingData (m_connected);
  }
  
}

// Retransmit timeout
void NaMPTCC::Retransmit (void)
{
  NS_LOG_FUNCTION (this);
  NS_LOG_LOGIC (this << " ReTxTimeout Expired at time " << Simulator::Now ().GetSeconds ());
  m_inFastRec = false;

  // If erroneous timeout in closed/timed-wait state, just return
  if (m_state == CLOSED || m_state == TIME_WAIT) return;
  // If all data are received, just return
  if (m_state <= ESTABLISHED && m_txBuffer.HeadSequence () >= m_highTxMark) return;

  // Notification
  bool hasDone = false;
  if (!m_notifyTimeout.IsNull ())
  {
    hasDone = m_notifyTimeout (this);
  }

  // default process
  if (!hasDone)
  {
    // According to RFC2581 sec.3.1, upon RTO, ssthresh is set to half of flight
    // size and cwnd is set to 1*MSS, then the lost packet is retransmitted and
    // TCP back to slow start
    m_ssThresh = std::max (2 * m_segmentSize, BytesInFlight () / 2);
    m_cWnd = m_segmentSize;
  }

  // common process
  m_nextTxSequence = m_txBuffer.HeadSequence (); // Restart from highest Ack
  NS_LOG_INFO ("RTO. Reset cwnd to " << m_cWnd <<
			 ", ssthresh to " << m_ssThresh << ", restart from seqnum " << m_nextTxSequence);
  m_rtt->IncreaseMultiplier (); 			// Double the next RTO
  DoRetransmit ();							// Retransmit the packet
}

void
NaMPTCC::SetSSThresh (uint32_t threshold)
{
  m_ssThresh = threshold;
}

uint32_t
NaMPTCC::GetSSThresh (void) const
{
  return m_ssThresh;
}

void
NaMPTCC::SetInitialCwnd (uint32_t cwnd)
{
  m_initialCWnd = cwnd;
}

uint32_t
NaMPTCC::GetInitialCwnd (void) const
{
  return m_initialCWnd;
}

void 
NaMPTCC::InitializeCwnd (void)
{
  /*
   * Initialize congestion window, default to 1 MSS (RFC2001, sec.1) and must
   * not be larger than 2 MSS (RFC2581, sec.3.1). Both m_initiaCWnd and
   * m_segmentSize are set by the attribute system in ns3::TcpSocket.
   */
  m_cWnd = m_initialCWnd * m_segmentSize;
}

} // namespace ns3
